import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Scale, Users, Home, Briefcase, FileText, User, Award, Shield, Clock, TrendingUp } from 'lucide-react'
import { motion } from 'framer-motion'
import './NewConsultation.css'

const NewConsultation = () => {
  const navigate = useNavigate()
  const [selectedArea, setSelectedArea] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const areas = [
    { 
      id: 'family', 
      name: 'Семейное право', 
      icon: Users,
      description: 'Разводы, алименты, опека'
    },
    { 
      id: 'labor', 
      name: 'Трудовое право', 
      icon: Briefcase,
      description: 'Трудовые споры, увольнения'
    },
    { 
      id: 'housing', 
      name: 'Жилищное право', 
      icon: Home,
      description: 'Сделки с недвижимостью'
    },
    { 
      id: 'criminal', 
      name: 'Уголовное право', 
      icon: FileText,
      description: 'Защита по уголовным делам'
    }
  ]

  const lawyers = [
    { name: 'Анна Соколова', experience: '15 лет', rating: 5, specialty: 'Семейное, Жилищное право' },
    { name: 'Михаил Петров', experience: '12 лет', rating: 5, specialty: 'Трудовое, Корпоративное право' },
    { name: 'Елена Волкова', experience: '18 лет', rating: 5, specialty: 'Уголовное, Административное право' },
    { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Жилищное, Семейное право' }
  ]

  const guarantees = [
    { icon: Shield, title: 'Конфиденциальность', text: 'Защита персональных данных' },
    { icon: Award, title: 'Профессионализм', text: 'Опытные специалисты' },
    { icon: Clock, title: 'Оперативность', text: 'Быстрые ответы' },
    { icon: TrendingUp, title: 'Результативность', text: '95% успешных дел' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedArea && selectedDate) {
      const savedConsultations = localStorage.getItem('lawConsultations')
      const consultations = savedConsultations ? JSON.parse(savedConsultations) : []
      
      const areaData = areas.find(a => a.id === selectedArea)
      const randomLawyer = lawyers[Math.floor(Math.random() * lawyers.length)]
      
      const newConsultation = {
        id: Date.now(),
        topic: areaData.name,
        lawyer: randomLawyer.name,
        status: 'В процессе',
        review: null
      }

      consultations.push(newConsultation)
      localStorage.setItem('lawConsultations', JSON.stringify(consultations))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedArea('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="new-consultation-page">
      <div className="container">
        <div className="consultation-header">
          <motion.h1 
            className="consultation-title"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6 }}
          >
            Новая консультация
          </motion.h1>
          <motion.p 
            className="consultation-subtitle"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            Выберите область права и дату
          </motion.p>
        </div>

        <motion.section 
          className="consultation-form-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <form className="consultation-form" onSubmit={handleSubmit}>
            <div className="form-group">
              <label className="form-label">
                <Scale size={18} strokeWidth={2} />
                Область права
              </label>
              <div className="areas-list">
                {areas.map((area) => {
                  const IconComponent = area.icon
                  return (
                    <motion.button
                      key={area.id}
                      type="button"
                      className={`area-item ${selectedArea === area.id ? 'selected' : ''}`}
                      onClick={() => setSelectedArea(area.id)}
                      whileHover={{ x: 5 }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <div className="area-item-icon">
                        <IconComponent size={24} strokeWidth={2} />
                      </div>
                      <div className="area-item-content">
                        <h3 className="area-item-name">{area.name}</h3>
                        <p className="area-item-desc">{area.description}</p>
                      </div>
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar size={18} strokeWidth={2} />
                Дата консультации
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <motion.button 
              type="submit" 
              className="submit-button" 
              disabled={!selectedArea || !selectedDate}
              whileHover={{ y: -2 }}
              whileTap={{ scale: 0.98 }}
            >
              Записаться
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={20} strokeWidth={2} />
                <span>Вы успешно записались на консультацию!</span>
              </motion.div>
            )}
          </form>
        </motion.section>

        <motion.section 
          className="lawyers-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title-minimal">Наши юристы</h2>
          <div className="lawyers-list">
            {lawyers.map((lawyer, index) => (
              <motion.div
                key={lawyer.name}
                className="lawyer-item"
                initial={{ opacity: 0, x: -30 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ x: 8, borderColor: 'var(--primary)' }}
              >
                <div className="lawyer-item-left">
                  <div className="lawyer-item-icon">
                    <User size={24} strokeWidth={2} />
                  </div>
                  <div className="lawyer-item-content">
                    <h3 className="lawyer-item-name">{lawyer.name}</h3>
                    <div className="lawyer-item-details">
                      <span className="lawyer-item-exp">{lawyer.experience} опыта</span>
                      <div className="lawyer-item-rating">
                        {Array.from({ length: 5 }, (_, i) => (
                          <Award key={i} size={14} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                        ))}
                      </div>
                    </div>
                    <p className="lawyer-item-specialty">{lawyer.specialty}</p>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="guarantees-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title-minimal">Гарантии</h2>
          <div className="guarantees-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-item"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ x: 5, borderColor: 'var(--primary)' }}
                >
                  <div className="guarantee-icon">
                    <IconComponent size={24} strokeWidth={2} />
                  </div>
                  <div className="guarantee-content">
                    <h3 className="guarantee-title">{guarantee.title}</h3>
                    <p className="guarantee-text">{guarantee.text}</p>
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default NewConsultation

